<?php
// submit_booking.php
require_once 'config.php';

// Set content type
header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

$response = ['success' => false, 'message' => ''];

try {
    // Store form data in session for success page
    $_SESSION['booking_data'] = [
        'organization' => $_POST['organization'] ?? '',
        'representativeName' => $_POST['representativeName'] ?? '',
        'email' => $_POST['email'] ?? '',
        'eventName' => $_POST['eventName'] ?? '',
        'eventDates' => $_POST['eventDates'] ?? '',
        'phone' => $_POST['phone'] ?? ''
    ];

    // If database connection exists, save to database
    if ($pdo) {
        $pdo->beginTransaction();

        // Handle signature upload
        $signaturePath = null;
        if (!empty($_POST['signatureData']) && $_POST['signatureData'] !== 'data:,') {
            $signaturePath = saveSignatureFromData($_POST['signatureData']);
        } elseif (!empty($_FILES['signatureUpload']['name'])) {
            $signaturePath = saveSignatureFromFile($_FILES['signatureUpload']);
        }

        // Insert main booking
        $stmt = $pdo->prepare("
            INSERT INTO bookings (
                organization_name, representative_name, phone, email, 
                event_name, event_description, event_dates, start_time, end_time,
                max_participants, elaborate_setup, music_event, music_details,
                vvip, vvip_names, other_info, declaration_name, signature_path
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");

        $stmt->execute([
            $_POST['organization'] ?? '',
            $_POST['representativeName'] ?? '',
            $_POST['phone'] ?? '',
            $_POST['email'] ?? '',
            $_POST['eventName'] ?? '',
            $_POST['eventDescription'] ?? '',
            $_POST['eventDates'] ?? '',
            $_POST['startTime'] ?? '',
            $_POST['endTime'] ?? '',
            $_POST['participants'] ?? 0,
            $_POST['elaborateSetup'] ?? '',
            $_POST['musicEvent'] ?? '',
            $_POST['musicDetails'] ?? '',
            $_POST['vvip'] ?? '',
            $_POST['vvipNames'] ?? '',
            $_POST['otherInfo'] ?? '',
            $_POST['declarationName'] ?? '',
            $signaturePath
        ]);

        $bookingId = $pdo->lastInsertId();

        // Store booking ID in session
        $_SESSION['booking_data']['booking_id'] = $bookingId;

        // Insert room selections
        foreach ($_POST as $key => $value) {
            if (strpos($key, 'room_') === 0 && !empty($value) && $value !== 'on') {
                $roomName = $value;
                
                $stmt = $pdo->prepare("
                    INSERT INTO booking_rooms (booking_id, room_name, room_dates) 
                    VALUES (?, ?, ?)
                ");
                $stmt->execute([$bookingId, $roomName, '']);
            }
        }

        // Insert event nature
        if (!empty($_POST['nature'])) {
            foreach ($_POST['nature'] as $nature) {
                $specify = ($nature === 'other') ? ($_POST['natureOtherSpecify'] ?? '') : '';
                
                $stmt = $pdo->prepare("
                    INSERT INTO booking_nature (booking_id, nature_type, nature_specify) 
                    VALUES (?, ?, ?)
                ");
                $stmt->execute([$bookingId, $nature, $specify]);
            }
        }

        $pdo->commit();
        
        $response['success'] = true;
        $response['message'] = 'Booking request submitted successfully!';
        $response['booking_id'] = $bookingId;
        
    } else {
        // Database not available, but form is still processed
        $response['success'] = true;
        $response['message'] = 'Booking request received! (Database temporarily unavailable)';
        $response['booking_id'] = 'temp-' . uniqid();
    }

    // Send confirmation email (optional)
    if (!empty($_POST['email'])) {
        sendConfirmationEmail($_POST);
    }

} catch (Exception $e) {
    if ($pdo) {
        $pdo->rollBack();
    }
    
    $response['success'] = false;
    $response['message'] = 'Error: ' . $e->getMessage();
    
    error_log("Booking submission error: " . $e->getMessage());
}

echo json_encode($response);

function saveSignatureFromData($signatureData) {
    if (empty($signatureData)) return null;
    
    $uploadDir = 'uploads/signatures/';
    
    // Remove data URL prefix
    $signatureData = str_replace('data:image/png;base64,', '', $signatureData);
    $signatureData = str_replace(' ', '+', $signatureData);
    
    // Generate unique filename
    $filename = 'signature_' . uniqid() . '.png';
    $filepath = $uploadDir . $filename;
    
    // Save file
    if (file_put_contents($filepath, base64_decode($signatureData))) {
        return $filepath;
    }
    
    return null;
}

function saveSignatureFromFile($file) {
    if (empty($file) || $file['error'] !== UPLOAD_ERR_OK) {
        return null;
    }
    
    $uploadDir = 'uploads/signatures/';
    $allowedTypes = ['image/jpeg', 'image/png', 'image/gif'];
    $maxSize = 5 * 1024 * 1024; // 5MB

    if (!in_array($file['type'], $allowedTypes)) {
        throw new Exception('Invalid file type. Please upload an image.');
    }

    if ($file['size'] > $maxSize) {
        throw new Exception('File too large. Maximum size is 5MB.');
    }

    $filename = 'signature_' . uniqid() . '_' . preg_replace('/[^a-zA-Z0-9._-]/', '_', $file['name']);
    $filepath = $uploadDir . $filename;

    if (move_uploaded_file($file['tmp_name'], $filepath)) {
        return $filepath;
    }

    throw new Exception('Failed to save uploaded file');
}

function sendConfirmationEmail($data) {
    $to = $data['email'] ?? '';
    if (empty($to)) return;
    
    $subject = 'CEDI Conference Centre - Booking Request Received';
    
    $message = "
    <html>
    <head>
        <style>
            body { font-family: Arial, sans-serif; line-height: 1.6; }
            .header { background: #0056b3; color: white; padding: 30px; text-align: center; }
            .content { padding: 30px; background: #f8f9fa; }
            .footer { background: #e9ecef; padding: 20px; text-align: center; }
            .details { background: white; padding: 20px; margin: 20px 0; border-radius: 8px; }
        </style>
    </head>
    <body>
        <div class='header'>
            <h1>CEDI CONFERENCE CENTRE</h1>
            <h2>Booking Request Confirmation</h2>
        </div>
        <div class='content'>
            <p>Dear " . ($data['representativeName'] ?? 'Valued Client') . ",</p>
            <p>Thank you for your booking request for <strong>" . ($data['eventName'] ?? 'your event') . "</strong>.</p>
            
            <div class='details'>
                <h3>Booking Summary:</h3>
                <p><strong>Organization:</strong> " . ($data['organization'] ?? '') . "</p>
                <p><strong>Event:</strong> " . ($data['eventName'] ?? '') . "</p>
                <p><strong>Dates:</strong> " . ($data['eventDates'] ?? '') . "</p>
                <p><strong>Contact:</strong> " . ($data['phone'] ?? '') . "</p>
            </div>
            
            <p>We have received your request and will review it shortly. You will hear from us within 2-3 business days.</p>
            <p>If you have any urgent enquiries, please don't hesitate to contact us.</p>
        </div>
        <div class='footer'>
            <h3>Contact Information</h3>
            <p><strong>Tel:</strong> 050 668 6685</p>
            <p><strong>WhatsApp:</strong> 050 668 6685</p>
            <p><strong>Email:</strong> ccc.econ@ug.edu.gh</p>
            <p><strong>Address:</strong> Department of Economics, University of Ghana</p>
        </div>
    </body>
    </html>
    ";

    $headers = "MIME-Version: 1.0" . "\r\n";
    $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
    $headers .= "From: CEDI Conference Centre <ccc.econ@ug.edu.gh>" . "\r\n";
    $headers .= "Reply-To: ccc.econ@ug.edu.gh" . "\r\n";

    @mail($to, $subject, $message, $headers);
}
?>