<?php
// create_admin.php
session_start();
require_once 'config.php';

// Check if user is logged in and is super admin
if (!isAdminLoggedIn() || !isSuperAdmin()) {
    header('Location: admin_login.php');
    exit;
}

$success = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = trim($_POST['username']);
    $password = $_POST['password'];
    $full_name = trim($_POST['full_name']);
    $email = trim($_POST['email']);
    $role = $_POST['role'];
    
    // Validate inputs
    if (empty($username) || empty($password) || empty($full_name) || empty($email)) {
        $error = "All fields are required!";
    } elseif (strlen($password) < 6) {
        $error = "Password must be at least 6 characters long!";
    } else {
        if (createAdmin($username, $password, $full_name, $email, $role)) {
            $success = "Admin user created successfully!";
            // Clear form
            $_POST = array();
        } else {
            $error = "Failed to create admin user. Username might already exist.";
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Create New Admin - CEDI Conference Centre</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary: #0056b3;
            --secondary: #f8f9fa;
            --success: #28a745;
            --danger: #dc3545;
            --warning: #ffc107;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }

        body {
            background: #f5f6f8;
            padding: 20px;
        }

        .container {
            max-width: 600px;
            margin: 0 auto;
            background: white;
            border-radius: 10px;
            box-shadow: 0 3px 10px rgba(0,0,0,0.1);
            padding: 30px;
        }

        .header {
            display: flex;
            align-items: center;
            gap: 15px;
            margin-bottom: 30px;
            padding-bottom: 15px;
            border-bottom: 2px solid #e9ecef;
        }

        .header i {
            font-size: 2rem;
            color: var(--primary);
        }

        .header h1 {
            color: var(--primary);
            font-size: 1.8rem;
        }

        .form-group {
            margin-bottom: 20px;
        }

        .form-group label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #555;
        }

        .form-control {
            width: 100%;
            padding: 12px;
            border: 2px solid #e9ecef;
            border-radius: 6px;
            font-size: 1rem;
            transition: border-color 0.3s ease;
        }

        .form-control:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(0, 86, 179, 0.1);
        }

        .btn {
            padding: 12px 25px;
            border: none;
            border-radius: 6px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }

        .btn-primary {
            background: var(--primary);
            color: white;
        }

        .btn-primary:hover {
            background: #004494;
            transform: translateY(-2px);
        }

        .btn-secondary {
            background: #6c757d;
            color: white;
        }

        .btn-secondary:hover {
            background: #545b62;
        }

        .alert {
            padding: 15px;
            border-radius: 6px;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .alert-success {
            background: #d4edda;
            color: #155724;
            border-left: 4px solid var(--success);
        }

        .alert-error {
            background: #f8d7da;
            color: #721c24;
            border-left: 4px solid var(--danger);
        }

        .form-actions {
            display: flex;
            gap: 15px;
            margin-top: 30px;
        }

        .password-strength {
            margin-top: 5px;
            font-size: 0.8rem;
            font-weight: 600;
        }

        .role-info {
            background: #e7f3ff;
            padding: 15px;
            border-radius: 6px;
            margin-top: 20px;
            font-size: 0.9rem;
            color: var(--primary);
        }

        .role-info h4 {
            margin-bottom: 8px;
            color: var(--primary);
        }

        .role-info ul {
            margin-left: 20px;
        }

        .role-info li {
            margin-bottom: 5px;
        }
    </style>
</head>
<body>
    <?php include 'admin_header.php'; ?>

    <div class="container">
        <div class="header">
            <i class="fas fa-user-plus"></i>
            <h1>Create New Admin</h1>
        </div>

        <?php if ($success): ?>
            <div class="alert alert-success">
                <i class="fas fa-check-circle"></i> <?php echo $success; ?>
            </div>
        <?php endif; ?>

        <?php if ($error): ?>
            <div class="alert alert-error">
                <i class="fas fa-exclamation-triangle"></i> <?php echo $error; ?>
            </div>
        <?php endif; ?>

        <form method="POST" action="">
            <div class="form-group">
                <label for="username">Username *</label>
                <input type="text" id="username" name="username" class="form-control" required 
                       value="<?php echo htmlspecialchars($_POST['username'] ?? ''); ?>"
                       placeholder="Enter unique username">
            </div>

            <div class="form-group">
                <label for="password">Password *</label>
                <input type="password" id="password" name="password" class="form-control" required 
                       placeholder="Minimum 6 characters" minlength="6">
                <div id="password-strength" class="password-strength"></div>
            </div>

            <div class="form-group">
                <label for="full_name">Full Name *</label>
                <input type="text" id="full_name" name="full_name" class="form-control" required 
                       value="<?php echo htmlspecialchars($_POST['full_name'] ?? ''); ?>"
                       placeholder="Enter full name">
            </div>

            <div class="form-group">
                <label for="email">Email Address *</label>
                <input type="email" id="email" name="email" class="form-control" required 
                       value="<?php echo htmlspecialchars($_POST['email'] ?? ''); ?>"
                       placeholder="Enter email address">
            </div>

            <div class="form-group">
                <label for="role">Role *</label>
                <select id="role" name="role" class="form-control" required>
                    <option value="admin" <?php echo ($_POST['role'] ?? '') === 'admin' ? 'selected' : ''; ?>>Admin</option>
                    <option value="super_admin" <?php echo ($_POST['role'] ?? '') === 'super_admin' ? 'selected' : ''; ?>>Super Admin</option>
                </select>
            </div>

            <div class="role-info">
                <h4><i class="fas fa-info-circle"></i> Role Permissions</h4>
                <ul>
                    <li><strong>Admin:</strong> Can manage bookings, view reports, and export data</li>
                    <li><strong>Super Admin:</strong> Full system access including admin management</li>
                </ul>
            </div>

            <div class="form-actions">
                <button type="submit" class="btn btn-primary">
                    <i class="fas fa-save"></i> Create Admin
                </button>
                <a href="admin_management.php" class="btn btn-secondary">
                    <i class="fas fa-arrow-left"></i> Back to Admin List
                </a>
            </div>
        </form>
    </div>

    <script>
        // Password strength indicator
        const passwordInput = document.getElementById('password');
        const passwordStrength = document.getElementById('password-strength');

        passwordInput.addEventListener('input', function() {
            const password = this.value;
            let strength = 'Weak';
            let color = '#dc3545';

            if (password.length >= 8) {
                strength = 'Good';
                color = '#ffc107';
            }
            if (password.length >= 12) {
                strength = 'Strong';
                color = '#28a745';
            }

            passwordStrength.textContent = `Password strength: ${strength}`;
            passwordStrength.style.color = color;
        });

        // Auto-generate username from full name
        document.getElementById('full_name').addEventListener('blur', function() {
            const fullName = this.value.trim();
            const usernameInput = document.getElementById('username');
            
            if (fullName && !usernameInput.value) {
                // Generate username from full name (first name + last name initial)
                const names = fullName.toLowerCase().split(' ');
                if (names.length >= 2) {
                    const username = names[0] + names[1].charAt(0);
                    usernameInput.value = username;
                } else if (names.length === 1) {
                    usernameInput.value = names[0].toLowerCase();
                }
            }
        });

        // Form validation
        document.querySelector('form').addEventListener('submit', function(e) {
            const password = document.getElementById('password').value;
            if (password.length < 6) {
                e.preventDefault();
                alert('Password must be at least 6 characters long!');
                return false;
            }
            
            const username = document.getElementById('username').value;
            if (username.length < 3) {
                e.preventDefault();
                alert('Username must be at least 3 characters long!');
                return false;
            }
        });
    </script>
</body>
</html>