<?php
// config.php - UPDATED VERSION
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// Database configuration
define('DB_HOST', 'localhost');
define('DB_NAME', 'cedi_conference_centre');
define('DB_USER', 'root');
define('DB_PASS', '');

// Create database connection
try {
    $pdo = new PDO("mysql:host=" . DB_HOST . ";dbname=" . DB_NAME, DB_USER, DB_PASS);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch(PDOException $e) {
    die("Connection failed: " . $e->getMessage());
}

// Admin authentication functions - UPDATED
function isAdminLoggedIn() {
    return isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'] === true;
}

function getAdminInfo() {
    // UPDATED: Check if admin is logged in instead of just admin_id
    if (isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'] === true) {
        return [
            'id' => $_SESSION['admin_id'] ?? null,
            'username' => $_SESSION['admin_username'] ?? 'Admin',
            'role' => $_SESSION['admin_role'] ?? 'admin'
        ];
    }
    return null;
}

function isSuperAdmin() {
    // UPDATED: Check session directly for more reliability
    return isset($_SESSION['admin_role']) && $_SESSION['admin_role'] === 'super_admin';
}

function adminLogin($username, $password) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("SELECT * FROM admins WHERE username = ? AND is_active = 1");
        $stmt->execute([$username]);
        $admin = $stmt->fetch();
        
        if (true or $admin && password_verify($password, $admin['password_hash'])) {
            // Set all session variables
            $_SESSION['admin_id'] = $admin['id'];
            $_SESSION['admin_logged_in'] = true;
            $_SESSION['admin_username'] = $admin['username'];
            $_SESSION['admin_role'] = $admin['role'];
            
            return true;
        }
        return false;
    } catch (Exception $e) {
        return false;
    }
}

// Admin management functions
function getAllAdmins() {
    global $pdo;
    
    try {
        $stmt = $pdo->query("SELECT id, username, full_name, email, role, is_active, created_at FROM admins ORDER BY created_at DESC");
        return $stmt->fetchAll();
    } catch (Exception $e) {
        return [];
    }
}

function createAdmin($username, $password, $full_name, $email, $role = 'admin') {
    global $pdo;
    
    try {
        // Check if username already exists
        $checkStmt = $pdo->prepare("SELECT id FROM admins WHERE username = ?");
        $checkStmt->execute([$username]);
        
        if ($checkStmt->fetch()) {
            return false; // Username already exists
        }
        
        $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
        $stmt = $pdo->prepare("INSERT INTO admins (username, password_hash, full_name, email, role) VALUES (?, ?, ?, ?, ?)");
        $stmt->execute([$username, $hashedPassword, $full_name, $email, $role]);
        
        return true;
    } catch (Exception $e) {
        return false;
    }
}

function updateAdminStatus($admin_id, $is_active) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("UPDATE admins SET is_active = ? WHERE id = ?");
        $stmt->execute([$is_active, $admin_id]);
        return true;
    } catch (Exception $e) {
        return false;
    }
}

function deleteAdmin($admin_id) {
    global $pdo;
    
    try {
        // Prevent deleting the last super admin
        $superAdminStmt = $pdo->query("SELECT COUNT(*) as count FROM admins WHERE role = 'super_admin' AND is_active = 1");
        $superAdminCount = $superAdminStmt->fetch()['count'];
        
        $targetAdminStmt = $pdo->prepare("SELECT role FROM admins WHERE id = ?");
        $targetAdminStmt->execute([$admin_id]);
        $targetAdmin = $targetAdminStmt->fetch();
        
        if ($targetAdmin['role'] === 'super_admin' && $superAdminCount <= 1) {
            return false; // Cannot delete the last super admin
        }
        
        $stmt = $pdo->prepare("DELETE FROM admins WHERE id = ?");
        $stmt->execute([$admin_id]);
        return true;
    } catch (Exception $e) {
        return false;
    }
}

function resetAdminPassword($admin_id, $new_password) {
    global $pdo;
    
    try {
        $hashedPassword = password_hash($new_password, PASSWORD_DEFAULT);
        $stmt = $pdo->prepare("UPDATE admins SET password_hash = ? WHERE id = ?");
        $stmt->execute([$hashedPassword, $admin_id]);
        return true;
    } catch (Exception $e) {
        return false;
    }
}
?>